import LoadingCircle from "@/components/icons/loading.circle";
import { Button } from "@/components/ui/button";
import { DialogClose, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { errorToStringFormatter } from "@/lib/formatters/error-formatters";
import { useRouter } from "@/lib/navigation.router";
import { ComponentWithClassName } from "@/types/utils";
import { zodResolver } from "@hookform/resolvers/zod";
import { signOut } from "aws-amplify/auth";
import { useState } from "react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { revokeUser } from "../lib/actions";
import { TransformedPendingUser, TransformedVerifiedUser } from "../type/transformers";

export function WorkflowDialogRevokeCurrentUser({ data, setClose, className }: ComponentWithClassName<{ data: NonNullable<TransformedVerifiedUser | TransformedPendingUser>, setClose: () => void }>) {
  const [errors, setErrors] = useState<(Error | string)[] | undefined>(undefined)
  const router = useRouter();
  const confirmation = data.displayName ?? "delete"

  const formSchema = z.object({
    confirmation_field: z.string()
      .refine((password) => password === confirmation, { message: `Form must match '${confirmation}'.` }),
  });

  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
  })
  const { formState: { isDirty, isValid, isSubmitting } } = form

  return (
    <DialogContent className={className}>
      <DialogHeader>
        <DialogTitle>Remove your account and permissions?</DialogTitle>
        <DialogDescription>
          This action is permanent. Your account&apos;s access to administrator and visitor permissions will be lost. To confirm this deletion type your email{" "}
          <span className="font-mono font-extrabold rounded-md bg-muted px-1.5 py-0.5">{confirmation}</span>{" "}
          in the form.
        </DialogDescription>
      </DialogHeader>

      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(async () => {
            const res = await revokeUser(data.displayName)
            if (!!res.errors && res.errors.length > 0) {
              setErrors(res.errors.map((e) => e.message))
              return;
            }

            await signOut()
            router.navigateAndReplaceHistory("/login")
            setClose()
          })}
          className="space-y-8"
        >

          <FormField
            control={form.control}
            name="confirmation_field"
            render={({ field }) => (
              <FormItem>
                <FormControl>
                  <Input
                    placeholder={confirmation}
                    type="text"
                    {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          {
            !!errors && (
              errors.map((e, i) => {
                return (
                  <div key={i} className="text-destructive text-xs" > {errorToStringFormatter(e)} </div>
                )
              })
            )
          }

          <div className="flex flex-row gap-2 justify-end">
            <DialogClose asChild>
              <Button variant={"secondary"}>Cancel</Button>
            </DialogClose>

            <Button type="submit" variant={"destructive"} disabled={!isDirty || !isValid || isSubmitting}>
              Revoke your access and logout{" "}
              {isSubmitting && <LoadingCircle className="ml-2" />}
            </Button>
          </div>
        </form>
      </Form>
    </DialogContent>
  )
}

